% Copyright (c) 2013, Massachusetts Institute of Technology
% This program was presented in the book "Visual Psychophysics:
% From Laboratory to Theory" by Zhong-Lin Lu and Barbara Dosher.
% The book is available at http://mitpress.mit.edu/books/visual-psychophysics

%%% Program ContrastSensitivityFunction.m

function TvCFunction (subjID, session)

%% Display Setup Module

% Define display parameters

whichScreen = max(Screen('screens'));
p.ScreenDistance = 30; 	% in inches
p.ScreenHeight = 15; 	% in inches
p.ScreenGamma = 2;	% from monitor calibration
p.maxLuminance = 100; % from monitor calibration
p.ScreenBackground = 0.5;

% Open the display window and hide the mouse cursor

if exist('onCleanup', 'class'), oC_Obj = onCleanup(@()sca); end % close any pre-existing PTB Screen window
%Prepare setup of imaging pipeline for onscreen window. 
PsychImaging('PrepareConfiguration'); % First step in starting pipeline
PsychImaging('AddTask', 'General', 'FloatingPoint32BitIfPossible');   % set up a 32-bit floatingpoint framebuffer
PsychImaging('AddTask', 'General', 'NormalizedHighresColorRange'); % normalize the color range ([0, 1] corresponds to [min, max])
PsychImaging('AddTask', 'General', 'EnablePseudoGrayOutput'); % enable high gray level resolution output with bitstealing
PsychImaging('AddTask', 'FinalFormatting', 'DisplayColorCorrection', 'SimpleGamma');  % setup Gamma correction method using simple power function for all color channels 
[windowPtr p.ScreenRect] = PsychImaging('OpenWindow', whichScreen, p.ScreenBackground);  % Finishes the setup phase for imaging pipeline, creates an onscreen window, performs all remaining configuration steps
PsychColorCorrection('SetEncodingGamma', windowPtr, 1 / p.ScreenGamma);  % set Gamma for all color channels
HideCursor;  % Hide the mouse cursor

% Get frame rate and set screen font

p.ScreenFrameRate = FrameRate(windowPtr);
Screen('TextFont', windowPtr, 'Times'); 
Screen('TextSize', windowPtr, 24);

%% Experimental Module

% Specify the stimulus
p.stimSize = 6;    % image diameter  in visual degrees
p.noises = [0 0.0156 0.0313 0.0625 0.125 0.165 0.250 0.33]; 
                   % external noise levels
p.contrasts = ...
     [0.0237 0.0395 0.0501 0.0639 0.0775 0.0872 0.1027
      0.0237 0.0395 0.0502 0.0640 0.0776 0.0872 0.1027
      0.0238 0.0397 0.0503 0.0642 0.0778 0.0875 0.1031
      0.0249 0.0415 0.0527 0.0672 0.0815 0.0917 0.1079
      0.0343 0.0572 0.0726 0.0925 0.1122 0.1262 0.1486
      0.0433 0.0723 0.0917 0.1169 0.1418 0.1595 0.1878
      0.0644 0.1074 0.1362 0.1737 0.2106 0.2369 0.2790
      0.0847 0.1412 0.1792 0.2284 0.2771 0.3117 0.3670];
p.nFactor = 3;     % noise pixel size
p.sf = 0.9;        % c/d
p.sigma = 1.1;     % degree
p.refAngle = 0;    % gabor reference angle
p.dAngle = 10;     % gabor tilt angle
repeats = 10;      % Number of trials to repeat for 
                   % each contrast
p.fixDuration = 0.25; 
p.ITI = 1;         % seconds between trials
keys = {'left' 'right' 'esc'};  % allowed response keys

% Compute stimulus parameters
ppd = pi/180 * p.ScreenDistance / p.ScreenHeight * ...
      p.ScreenRect(4);          % pixels per degree
m = round(p.stimSize * ppd /p.nFactor) * p.nFactor; 
                   % stimulus size in pixels
sf = p.sf / ppd;   % cycles per pixel
sc = round(p.sigma * ppd);      % sigma in pixels
fixXY = [[[-1 1] * 8 0 0] + p.ScreenRect(3) / 2;
        [0 0 [-1 1] * 8] + p.ScreenRect(4) / 2];
nContrast = size(p.contrasts, 2); 
                   % number of contrast levels
nNoise = numel(p.noises); % number of external noise levels
nTrials = repeats * nContrast * nNoise; 
                   % total number of trials

if nargin < 1, subjID = 's01'; end
if nargin < 2, session = 1; end
if session > 4
    fileName = sprintf('TvC_rst_%s_%02.0f.mat', subjID, session - 4);
    S = load(fileName, 'p');
    p.randSeed = ClockRandSeed(S.p.randSeed); 
        % use seed from session-4 session
else
    p.randSeed = ClockRandSeed;  
        % use clock to set seed for the random 
        % number generator
end

% procedural gabor allows us to change its parameters 
% very quickly
tex = CreateProceduralGabor(windowPtr, m, m, 0, ...
      [1 1 1 0] * 0.5, 1, 0.5);
noiseTex = zeros(1, 4);     % preallocate noise texture

% Initialize a table to set up experimental conditions
p.recLabel = {'trialIndex' 'contrastIndex' 'noiseIndex' ...
      'tileRight' 'respCorrect' 'respTime'};
rec = nan(nTrials, length(p.recLabel)); 
       % matrix rec is made of nTrials x 6 of NaN
rec(:, 1) = 1 : nTrials; % trial number from 1 to nTrials
contrastIndex = repmat(1 : nContrast, [nNoise 1 repeats]);
noiseIndex = repmat((1 : nNoise)', [1 nContrast repeats]);
tileRight = zeros(nNoise, nContrast, repeats) * 2; 
       % first set all to 0
tileRight(:, :, 1 : repeats / 2) = 1; 
       % change first half to 1
[rec(:, 2) ind] = Shuffle(contrastIndex(:)); 
       % shuffle contrast index
rec(:, 3) = noiseIndex(ind); 
       % shuffle noise index in the same order
rec(:, 4) = tileRight(ind);  
       % shuffle interval in the same order

% Prioritize display to optimize display timing
Priority(MaxPriority(windowPtr));

% Start experiment with instructions
str = ['Press left and right arrow keys for top left and '...
       'right tilt responses.\n\n''Press SPACE to ' ...
       'proceed.'];
DrawFormattedText(windowPtr, str, 'center', 'center', 1);
        % Draw Instruction text string centered in window
Screen('Flip', windowPtr);  
        % flip the text image into active buffer
Beeper;

WaitTill('space');      % wait till space bar is pressed
p.start = datestr(now); % record start time

% Run nTrials trials
for i = 1 : nTrials
    % parameters for this trial
    con = p.contrasts(rec(i, 3), rec(i, 2));
    noise = p.noises(rec(i, 3));
    ang = p.refAngle - sign(rec(i, 4) - 0.5) * p.dAngle;
    
    % Make 4 independent noise textures
    for j = 1 : 4
        gn = randn(m / p.nFactor);  % Gaussian noise
        while 1
            out = abs(gn) > 3; 
            nout = sum(out(:));
            if nout == 0, break; end
            gn(out) = randn(nout, 1);
        end
        gn = gn / 2 * noise + 0.5;  % [0 1]
        gn = Expand(gn, p.nFactor);
        noiseTex(j) = Screen('MakeTexture', windowPtr, ...
            gn, 0, 0, 2);
    end
    
    if i > 1
        if WaitTill(Secs + p.ITI, 'esc'), break; end 
            % wait for ITI
        if strcmp(key, 'esc'), break; end % to stop
    end
    
    Screen('DrawLines', windowPtr, fixXY, 3, 0); % fixation
    t0 = Screen('Flip', windowPtr);
    Screen('Flip', windowPtr, t0 + p.fixDuration);
    
    for j = 1 : 2  % 2 noise frames
        Screen('DrawTexture', windowPtr, noiseTex(j));
        Screen('Flip', windowPtr);
    end

     % 1 signal frame
    Screen('DrawTexture', windowPtr, tex, [], [], ang, ...
           [], [], [], [], 2, [180 sf sc con 1 0 0 0]);
    Screen('Flip', windowPtr);
    
    for j = 3 : 4  % 2 noise frames
        Screen('DrawTexture', windowPtr, noiseTex(j));
        Screen('Flip', windowPtr);
    end
    Screen('Flip', windowPtr);     % turn off last frame
       
    [key Secs] = WaitTill(keys);   % wait till response
    if iscellstr(key), key = key{1}; end 
        % take the first in case of multiple key presses
    if strcmp(key, 'esc'), break; end    % to stop
    rec(i, 5) = strcmp(key, 'right') == rec(i, 4); 
        % record correctness
    rec(i, 6) = Secs - t0;               % record respTime
    if rec(i, 5), Beeper; end            % beep if correct
end
p.finish = datestr(now);       % record finish time
fileName = sprintf('TvC_rst_%s_%02.0f.mat', subjID, ...
           session);
save(fileName, 'rec', 'p');    % save results


%% System Reinstatement Module

Priority(0);  % restore priority
sca; % close window and textures, restore color lookup table

